defmodule Reseau.Messager do
  @moduledoc """
  Une simple tâche contenant un messager, chargé d'écouter sur un socket.

  Le messager est une tâche simple : elle a besoin de connaître le processus
  à contacter en cas de nouveau message et le socket à écouter. Une fois
  un message reçu sur le socket, elle le transmet au processus.
  """

  use Task

  require Logger

  def start_link(client, socket) do
    Task.start_link(__MODULE__, :démarrer, [client, socket])
  end

  @doc """
  Lancement de la tâche.
  """
  @spec démarrer(pid(), port()) :: no_return()
  def démarrer(client, socket) do
    Logger.info("Une tâche a été créée pour écouter le client #{inspect(socket)}.")
    écouter(client, socket)
  end

  defp écouter(client, socket) do
    recevoir(client, socket)
    écouter(client, socket)
  end

  defp recevoir(client, socket) do
    :gen_tcp.recv(socket, 0)
    |> informer_client(client, socket)
  end

  defp informer_client({:ok, message}, client, _socket) do
    GenServer.cast(client, {:message, message})
  end

  defp informer_client(_, _client, socket) do
    Logger.info("Le client #{inspect(socket)} est à présent déconnecté.")
    exit(:shutdown)
  end
end
